/**
 * BotFake Recorder - Popup Script
 * Controls the recording session and updates UI
 */

// State
let isRecording = false;
let actionsCount = 0;

// DOM Elements
const statusDot = document.getElementById('statusDot');
const statusText = document.getElementById('statusText');
const actionsCountEl = document.getElementById('actionsCount');
const btnStart = document.getElementById('btnStart');
const btnStop = document.getElementById('btnStop');
const btnDownload = document.getElementById('btnDownload');
const btnCopy = document.getElementById('btnCopy');
const infoBox = document.getElementById('infoBox');

// Initialize
document.addEventListener('DOMContentLoaded', async () => {
  // Get current state from storage
  const state = await chrome.storage.local.get(['isRecording', 'actionsCount']);
  isRecording = state.isRecording || false;
  actionsCount = state.actionsCount || 0;
  updateUI();
});

// Event listeners
btnStart.addEventListener('click', startRecording);
btnStop.addEventListener('click', stopRecording);
btnDownload.addEventListener('click', downloadScenario);
btnCopy.addEventListener('click', copyScenario);

// Listen for updates from content script
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  if (message.type === 'RECORDING_UPDATE') {
    actionsCount = message.actionsCount;
    updateUI();
  }
});

async function startRecording() {
  // Get active tab
  const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
  
  if (!tab) {
    showError('No active tab found');
    return;
  }
  
  // Check if we can inject into this page
  if (tab.url.startsWith('chrome://') || tab.url.startsWith('edge://') || tab.url.startsWith('about:')) {
    showError('Cannot record on browser internal pages');
    return;
  }
  
  // Send message to content script to start recording
  try {
    await chrome.tabs.sendMessage(tab.id, { type: 'START_RECORDING' });
    
    isRecording = true;
    actionsCount = 0;
    await chrome.storage.local.set({ isRecording: true, actionsCount: 0, tabId: tab.id });
    updateUI();
  } catch (error) {
    // Content script not loaded, inject it
    try {
      await chrome.scripting.executeScript({
        target: { tabId: tab.id },
        files: ['recorder.js']
      });
      
      // Wait a bit and try again
      setTimeout(async () => {
        await chrome.tabs.sendMessage(tab.id, { type: 'START_RECORDING' });
        isRecording = true;
        actionsCount = 0;
        await chrome.storage.local.set({ isRecording: true, actionsCount: 0, tabId: tab.id });
        updateUI();
      }, 100);
    } catch (injectError) {
      showError('Failed to start recording: ' + injectError.message);
    }
  }
}

async function stopRecording() {
  const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
  
  if (tab) {
    try {
      await chrome.tabs.sendMessage(tab.id, { type: 'STOP_RECORDING' });
    } catch (e) {
      console.error('Failed to send stop message:', e);
    }
  }
  
  isRecording = false;
  await chrome.storage.local.set({ isRecording: false });
  updateUI();
}

async function downloadScenario() {
  const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
  
  if (tab) {
    try {
      chrome.tabs.sendMessage(tab.id, { type: 'DOWNLOAD_SCENARIO' });
    } catch (e) {
      showError('Failed to download scenario');
    }
  }
}

async function copyScenario() {
  const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
  
  if (tab) {
    try {
      const response = await chrome.tabs.sendMessage(tab.id, { type: 'GET_SCENARIO' });
      
      if (response && response.scenario) {
        await navigator.clipboard.writeText(JSON.stringify(response.scenario, null, 2));
        btnCopy.textContent = '✓ Copied!';
        setTimeout(() => {
          btnCopy.textContent = '📋 Copy';
        }, 2000);
      }
    } catch (e) {
      showError('Failed to copy scenario');
    }
  }
}

function updateUI() {
  if (isRecording) {
    statusDot.className = 'status-dot recording';
    statusText.textContent = 'Recording...';
    btnStart.disabled = true;
    btnStop.disabled = false;
    btnDownload.disabled = true;
    btnCopy.disabled = true;
    infoBox.innerHTML = '<strong>Recording in progress!</strong><br>Perform actions on the page, then click "Stop Recording".';
  } else {
    statusDot.className = 'status-dot' + (actionsCount > 0 ? ' ready' : '');
    statusText.textContent = actionsCount > 0 ? 'Stopped' : 'Ready';
    btnStart.disabled = false;
    btnStart.textContent = actionsCount > 0 ? '⏺ New Recording' : '⏺ Start Recording';
    btnStop.disabled = true;
    btnDownload.disabled = actionsCount === 0;
    btnCopy.disabled = actionsCount === 0;
    
    if (actionsCount > 0) {
      infoBox.innerHTML = `<strong>Recording complete!</strong><br>${actionsCount} actions recorded. Download or copy the JSON scenario.`;
    } else {
      infoBox.innerHTML = '<strong>How to use:</strong><br>1. Click "Start Recording"<br>2. Perform actions on the page<br>3. Click "Stop Recording"<br>4. Download or copy the JSON scenario';
    }
  }
  
  actionsCountEl.textContent = `${actionsCount} action${actionsCount !== 1 ? 's' : ''}`;
}

function showError(message) {
  infoBox.innerHTML = `<strong style="color: #ff4757;">Error:</strong><br>${message}`;
}

